using System;
using System.Drawing;
using System.Drawing.Imaging;
using System.Runtime.InteropServices;
using System.Threading;
using VNC.RFBDrawing;
using VNC.RFBDrawing.PixelDecoders;
using System.Collections;
using VNC.RFBProtocolHandling;

#region

// author: Dominic Ullmann, dominic_ullmann@swissonline.ch
// Version: 1.02
      
// VNC-Client for .NET
// Copyright (C) 2002 Dominic Ullmann

// author: Noh Seoung Hyun, gerranwizard@hotmail.com
// Version: 1.0

// VNC-Client for .NET Compact Framework
// Copyright (C) 2003 Noh Seoung Hyun

// This program is free software; 
// you can redistribute is and/or modify it under the terms of 
// the GNU General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.

#endregion

// This namespace contains classes that support drawing.

// This classes make the code in the RFB-Surface independent of the used Graphics library

namespace DrawingSupport
{
	/// <summary>
	/// The DrawSupport class is the main class used by the RFB-Surface to access 
	/// the drawing functionality
	/// </summary>
	public class DrawSupport
	{
		private OffScreenBuffer buffer;
		private OffScreenBuffer doubleBuffer;
		private Bitmap backBufferBitmap;
		/// <summary> The pixelformat of the remote frame buffer </summary>
		private VNC.RFBProtocolHandling.PixelFormat format;
		/// <summary> The width of the remote frame buffer </summary>
		private int width;
		/// <summary> The height of the remote frame buffer </summary>
		private int height;
		/// <summary> The depth of the remote frame buffer </summary>
		private int depth;
		/// <summary> is dispose already called </summary>
		protected bool disposed = false;

		/// <summary> constructor for DrawSupport </summary>
		public DrawSupport(int width, int height, int depth)
		{
			initialize(width, height, depth);
		}

		/// <summary> 
		/// register a view, which this DrawSupport instance draws to.
		/// In this step the view is prepared for drawing to it
		/// </summary>
		public void registerView(RFBView view)
		{
			// createing graphics-object for view
			Graphics viewGraphics = view.CreateGraphics();
		}

		/// <summary> initialize the DrawSupport </summary>
		public void initialize(int width, int height, int depth)
		{
			this.width = width;
			this.height = height;
			this.depth = depth;
			VNC.RFBProtocolHandling.PixelFormat tmpFormat = new VNC.RFBProtocolHandling.PixelFormat();
			this.format = tmpFormat;
			buffer = (OffScreenBuffer)getOffScreenBuffer(width, height);
			backBufferBitmap = buffer.getBitmap();
			doubleBuffer = (OffScreenBuffer)getOffScreenBuffer(width, height);
		}

		/// <summary> draws the specified region of backbuffer to screen </summary>
		public void drawBackBufferToScreen(RFBView screen, Rectangle region)
		{
			Monitor.Enter(buffer);
			
			// parameter region defines the changed part of the RFBSurface
			// regionOnScreen specifies the visible region of the RFBSurface
			Rectangle regionOnScreen = screen.ShowedRectangle;
			Rectangle intersection = Rectangle.Intersect(region, regionOnScreen);

			// nothing visible updated

			if((intersection.Width <= 0) || (intersection.Height <= 0))
			{
				Monitor.Exit(buffer); 
				return;
			}

			// the intersection definies the visible part of the region in RFBSurface-coordinates
			// identify destination coordinates in screen coordinates
			// the destination
			Rectangle regionDest = new Rectangle();
			// intersection.X - regionOnScreen.X lies in the visible area
			regionDest.X = (intersection.X - regionOnScreen.X);
			// intersection.Y - regionOnScreen.Y lies in the visible area
			regionDest.Y = (intersection.Y - regionOnScreen.Y);
			regionDest.Width = intersection.Width;
			regionDest.Height = intersection.Height;                        
                        
			// create Graphics object
			Graphics screenGraphics = screen.CreateGraphics();
			screenGraphics.DrawImage(backBufferBitmap, regionDest, intersection, GraphicsUnit.Pixel);
			screenGraphics.Dispose();

			Monitor.Exit(buffer);
		}

		/// <summary> draws the whole backbuffer to screen </summary>
		public void drawBackBufferToScreen(RFBView screen)
		{
			drawBackBufferToScreen(screen, new Rectangle(0, 0, width, height));
		}

		/// <summary> get DrawingObject for updating the backbuffer </summary>
		public DrawingObject getDrawingObject(int x, int y, int width, int height)
		{
			return new DrawingObject(x, y, width, height, doubleBuffer, this);
		}

		/// <summary> get an OffScreenBuffer with the given dimension </summary>
		public OffScreenBuffer getOffScreenBuffer(int width, int height)
		{
			return new OffScreenBuffer(width, height, depth);
		}

		/// <summary> creates an OffScreenBuffer with a part of the backbuffer as contents </summary>
		public OffScreenBuffer copyFromBackBuffer(int srcX, int srcY, int width, int height)
		{
			Monitor.Enter(buffer);
			OffScreenBuffer dst = getOffScreenBuffer(width,height);
			dst.drawOffScreenBuffer(buffer, 0, 0, width, height, srcX, srcY);
			Monitor.Exit(buffer);
			return dst;
		}

		/// <summary> 
		/// called by a DrawingObject to inform DrawSupport of a performed update.
		/// should only be called by a DrawingObject created by the getDrawingIbject method.
		/// </summary>
		protected internal void drawingObjectDone(int x, int y, int width, int height)
		{
			Monitor.Enter(buffer);
			buffer.drawOffScreenBuffer(doubleBuffer, x, y, width, height, x, y);
			Monitor.Exit(buffer);
		}

		/// <summary> 
		/// for freeing resources, call if DrawingSupport object is no longer needed,
		/// (is othrewise called during finalization)
		/// </summary>
		public void Dispose()
		{
			// waiting for last drawing operation to complete
			Monitor.Enter(buffer);
			buffer.Dispose();
			doubleBuffer.Dispose();
			disposed = true;
			Monitor.Exit(buffer);
		}        

		/// <summary> finalizer </summary>
		~DrawSupport()
		{
			if (!disposed)
			{
				Dispose(); 
			}
		}
	}

	/// <summary>
	/// provides functionality to draw to a buffer / to the screen.
	/// /<summary>
	/// <remarks>
	/// This class is a general implementation of DrawingObject,
	/// usable with many DrawingSupport classes
	/// </remakrs>
	public class DrawingObject
	{
		private int width;
		private int height;
		private int x;
		private int y;
		private OffScreenBuffer backBuffer;
		private DrawSupport draw;

		public int Width { get { return        width; } }
		public int Height { get { return height; } }
		public int X { get { return x; } }
		public int Y { get { return y; } }

		/// <summary> creates a DrawingObject, for drawing to a specified resion </summary>
		/// <remarks> 
		/// This constructor should only be used from methods/constructors within 
		/// a DrawingSupport class
		/// </remarks>
		public DrawingObject(int x, int y, int width, int height, OffScreenBuffer buffer, DrawSupport drawSup)
		{
			this.x = x;
			this.y = y;
			this.width = width;
			this.height = height;
			this.backBuffer = buffer;
			this.draw = drawSup;
		}
		
		/// <summary>
		/// This methods draws a fiiled rectangle in the specified color and with
		/// the specified size at the specified position
		/// </summary>
		public virtual void drawFilledRectangle(Color color, int x, int y, int width, int height)
		{
		backBuffer.drawFilledRectangle(color, this.x + x, this.y + y, width, height);
		}
	
		/// <summary> draw from the contents of an OffScreenBuffer </summary>
		public virtual void drawOffScreenBuffer(OffScreenBuffer src, int destX, int destY)
		{
			backBuffer.drawOffScreenBuffer(src, this.x + destX, this.y + destY);
		}

		/// <summary> This methods draws data from a pixel array. </summary>
		public virtual void drawFromPixelArray(
			Bitmap inputBuffer, int width, int height, int destX, int destY, PixelDecoder usedDecoder)
		{
			backBuffer.drawFromPixelArray(inputBuffer, width, height, this.x + destX,
				this.y + destY, usedDecoder);
		}
		
		/// <summary> 
		/// calling this to tell update is done, forcing the changes made with
		/// this DrawingObject to appear
		/// </summary>
		public virtual void updateDone()
		{
			draw.drawingObjectDone(x, y, width, height);
		}
	}

	/// <summary> a buffer for buffering bitmap-data </summary>
	public class OffScreenBuffer
	{
		/// <summary>
		/// The width of the OffScreenBuffer
		/// </summary>
		private int width;
		/// <summary>
		/// The height of the OffScreenBuffer
		/// </summary>
		private int height;
		private Bitmap buffer;
		private SolidBrush brush;
		private Graphics graphics;
		private bool disposed = false;
		private int depth;
                    
		int Width { get { return width; } }
		int Height { get { return height; } }
		internal Bitmap getBitmap() { return buffer; }
		/// <summary> constructor for OffScreenBuffer, uses an existing Bitmap to store it's data </summary>
		/// <param name="bitmap">where the data of the offscreenbuffer is located</param>        
		/// <param name="depth"></param>
		public OffScreenBuffer(Bitmap bitmap, int depth)
		{
			buffer = bitmap;
			width = bitmap.Width;
			height = bitmap.Height;
			this.depth = depth;
			brush = new SolidBrush(new Color());
			graphics = Graphics.FromImage(bitmap);
		}
		
		/// <summary> constructor for DotNetOffScreenBuffer, creates a new Bitmap to store it's data </summary>
		/// <param name="width">the width of the buffer</param>        
		/// <param name="height">the height of the buffer</param>
		/// <param name="format">the pixelformat for the buffer</param>
		/// <param name="depth"></param>
		public OffScreenBuffer(int width, int height,int depth) 
		{
			this.width = width;
			this.height = height;
			this.depth = depth;
			buffer = new Bitmap(width, height);
			graphics = Graphics.FromImage(buffer);
			brush = new SolidBrush(new Color());

	}
		
		/// <summary> draws a filled rectangle to the OffScreenBuffer </summary>
		public void drawFilledRectangle(Color color, int x, int y, int width, int height)
		{
			brush.Color = color;
			graphics.FillRectangle(brush, x, y, width, height); 
		}

		/// <summary> draws the contents of another OffScreenBuffer to the OffScreenBuffer </summary>
		public void drawOffScreenBuffer(OffScreenBuffer src, int destX, int destY)
		{
			graphics.DrawImage(((OffScreenBuffer)src).getBitmap(), destX, destY);
		}

		/// <summary> draws the contents of another OfferScreenBuffer to the OffScreenBuffer </summary>
		public void drawOffScreenBuffer(
			OffScreenBuffer src, int destX, int destY, int width, int height, int srcX, int srcY)
		{
			graphics.DrawImage(((OffScreenBuffer)src).getBitmap(), destX, destY, new Rectangle(srcX, srcY, width, height), GraphicsUnit.Pixel);
		}

		/// <summary> draw text to the OffScreenBuffer </summary>
		public void drawText(string text, int x, int y, Font font, Color color) 
		{
			brush.Color = color;
			graphics.DrawString(text, font, brush, x, y);
		}                       
		
		/// <summary> draw the data from a pixel array to the OffScreenBuffer </summary>
		public void drawFromPixelArray(Bitmap bitmap, int width, int height, int destX, int destY, PixelDecoder usedDecoder) 
		{
			graphics.DrawImage(bitmap, destX, destY);
		}                        

		/// <summary> disposes resources allocated by offscreenbuffer </summary>
		public void Dispose() 
		{
			brush.Dispose();
			graphics.Dispose();
		buffer.Dispose();
		disposed = true;
		}                

		/// <summary> finalizer </summary>
		~OffScreenBuffer() 
		{
			if (!disposed) { Dispose(); }
		}
	}
}