/*  -
   Copyright (C) 2006 Weongyo Jeong (weongyo@gmail.com)

This file is part of ROVM.

ROVM is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

ROVM is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA
02111-1307, USA.  */

#include <stdio.h>

#include "types.h"
#include "mpool.h"
#include "tktree.h"
#include "clstree.h"
#include "sha1.h"
#include "listen.h"
#include "rovm.h"

#include "thread.h"
#include "thread_mutex.h"
#include "thread_cond.h"

#include "mpm_worker_fdqueue.h"
#include "mpm_worker_pod.h"
#include "mpm_worker.h"

#include "connection.h"
#include "request.h"

#include "log.h"
#include "utils.h"
#include "ticket.h"

/**
   TKTREE  ϳ node key    ȣǴ callback Լ.

   @param k	 node key
 */
void
rovm_tktree_delete_key (k)
     tktree_key k;
{
  char *tid = (char *) k;

  if (tid)
    rc_free (tid);
}

/**
   TKTREE  ϳ node value    ȣǴ callback Լ.

   @param v	 node value
 */
void
rovm_tktree_delete_value (v)
     tktree_value v;
{
  struct ticket *tk = (struct ticket *) v;

  destroy_ticket (tk);
}

/*   ticket   tree function  wrapper  Ǿ ֽϴ.  
   ̷ ϴ    ׽Ʈ ̸ܰ, ̸ ְ tree  
     Ͽ Դϴ.  */

/**
   tktree SP  key  value  Ѵ.

   @note	Wrapper Function Դϴ.

   @param sp	 ticket tree
   @param key	 key 
   @param value  value 
   @return	   Ǿٸ 0 , ׷ ʴٸ -1  ȯѴ.
 */
int
tktree_insert (sp, key, value)
     tktree_t *sp;
     char *key;
     void *value;
{
  splay_tree_node_t n;

  if (!sp || !key)
    return -1;

  n = splay_tree_insert (sp, (splay_tree_key) key, (splay_tree_value) value);
  if (!n)
    return -1;

  return 0;
}

/**
   tktree SP  KEY   key  value  ** Ѵ.

   @param sp    Ticket Tree 
   @param key    Ticket ID
 */
void
tktree_remove (sp, key)
     tktree_t *sp;
     char *key;
{
  splay_tree_key backup;
  splay_tree_node_t n = splay_tree_lookup (sp, (tktree_key) key);

  if (!n)
    return;

  backup = n->key;
  /* !! splay_tree_remove () Լ Node  Value   key 
      ʱ , ̸ Ʒ key  ش.  */
  splay_tree_remove (sp, (splay_tree_key) n->key);
  rovm_tktree_delete_key (backup);
}

/**
   tktree SP  key  KEY  node  ã´.     ,
   NULL  ȯϸ, ߰Ͽ , ش value  ȯѴ.

   @note	Wrapper Function Դϴ.

   @param sp	 ticket tree
   @param key	˻  key 
   @return	߰ key  value  ȯѴ.   ʴ ٸ, NULL 
                ȯѴ.
 */
void *
tktree_lookup (sp, key)
     tktree_t *sp;
     char *key;
{
  splay_tree_node_t n = splay_tree_lookup (sp, (tktree_key) key);

  if (!n)
    return NULL;

  return (void *) n->value;
}

/**
   Ticket Tree   node  walking   մϴ.

   @param sp    Ticket Tree 
   @param fn     node   ȣǴ function.
   @param data   fn  ѱ data
   @return        fn    ȯǴ  0  ƴ ,
                walking  ߰ ٷ ش  ȯϰ ˴ϴ.
 */
int
tktree_walking (sp, fn, data)
     tktree_t *sp;
     tktree_foreach_fn fn;
     void *data;
{
  return splay_tree_foreach (sp, fn, data);
}

/**
   ο ticket tree  Ѵ.

   @note	Wrapper Function Դϴ.

   @param compare_fn		Tree  compare function
   @param delete_key_fn		Tree  key     Լ.
   @param delete_value_fn	Tree  value     Լ.
   @return   ticket tree ȯѴ.
 */
tktree_t *
init_tktree (compare_fn, delete_key_fn, delete_value_fn)
     tktree_compare_fn compare_fn;
     tktree_delete_key_fn delete_key_fn;
     tktree_delete_value_fn delete_value_fn;
{
  return (tktree_t *) splay_tree_new (compare_fn, delete_key_fn, delete_value_fn);
}


/**
    ̻ ticket tree  ʿ , ش ׵    Ѵ.

   @note	Wrapper Function Դϴ.

   @param tk	 tktree_t ü.
 */
void
finish_tktree (tk)
     tktree_t *tk;
{
  splay_tree_delete (tk);
}

/**
   Ticket Key   Ͽ   , 0  ȯϰ ׷  
   1 Ȥ -1  ȯѴ.

   @param k1	Key 1 
   @param k2	Key 2 
*/
static inline int
compare_ticketvalue (t1, t2)
     char *t1, *t2;
{
  int __tmpi;

  for (__tmpi=0; __tmpi<TICKETID_SIZE; __tmpi++)
    {
      if ((int) t1[__tmpi] < (int) t2[__tmpi])
        return -1;
      else if ((int) t1[__tmpi] > (int) t2[__tmpi])
        return 1;
    }

  return 0;
}

/**
   Splay Tree  ƾ.  K1  K2   , 0  ȯѴ.  ׷  ,
   -1 Ȥ 1  ȯȴ.

   @param k1 Key 1 
   @param k2 Key 2 
   @return  K1  K2    0  ȯѴ.
*/
int
rovm_tktree_compare (k1, k2)
     tktree_key k1;
     tktree_key k2;
{
  return compare_ticketvalue ((char *) k1, (char *) k2);
}

/**
   ROVM ü tktree  ʱȭմϴ.

   @param r	ROVM ü 
   @return	 Ǿ  0 , ׷ ʴٸ -1  ȯѴ.
 */
int
init_rovm_tktree (r)
     struct rovm *r;
{
  rc_status_t rv;

  ROVM_TKTREE (r) = init_tktree ((tktree_compare_fn) rovm_tktree_compare,
				 (tktree_delete_key_fn) rovm_tktree_delete_key,
				 (tktree_delete_value_fn) rovm_tktree_delete_value);
  if (!ROVM_TKTREE (r))
    return -1;

  rv = rc_thread_mutex_create (&ROVM_TKMUTEX (r), RC_THREAD_MUTEX_DEFAULT, ROVM_POOL (r));
  if (rv != RC_SUCCESS)
    return -1;

  return 0;
}

